-- Localizable strings
local LocStr = {
	mm			= _("mm"),
	PodZones	= _("Pod Zones"),
	S8TsM		= _("S-8TsM"),
	Type		= _("Type"),
	UnGdRkts	= _("UnGd Rkts"),
}

-- Rocket launchers' parameters
local RLData = {
	B8M1 = {
		attribute			= {wsType_Weapon, wsType_NURS, wsType_Container, WSTYPE_PLACEHOLDER},
		category			= CAT_ROCKETS,
		connNamePattern		= "tube_%d",
		Count				= 20,
		Cx_pil				= 0.00146484375,
		displayNamePrfx		= ("%s - 20 x %s, 80 %s"):format(_("B-8M1"), LocStr.UnGdRkts, LocStr.mm),
		Elements			= ContainerWithSubshapes("B-20", "s-8_kom", 20, "tube_%d"),
		modelFileName		= "B-20",
		PictureBlendColor	= "0xffffffff",
		Picture				= "rus_B-8M1_S-8KOM.png",
		Weight_Empty		= 137.5,
	},
	B8V20A = {
		attribute			= {wsType_Weapon, wsType_NURS, wsType_Container, WSTYPE_PLACEHOLDER},
		category			= CAT_ROCKETS,
		connNamePattern		= "tube_%d",
		Count				= 20,
		Cx_pil				= 0.00213134765625,
		displayNamePrfx		= ("%s - 20 x %s, 80 %s"):format(_("B-8V20A"), LocStr.UnGdRkts, LocStr.mm),
		Elements			= ContainerWithSubshapes("B-8V20A", "s-8_kom", 20, "tube_%d"),
		modelFileName		= "B-8V20A",
		Picture				= "rus_B-8V20A_S-8KOM.png",
		PictureBlendColor	= "0xffffffff",
		Weight_Empty		= 123,
	},
	LR25 = {
		attribute			= {wsType_Weapon, wsType_NURS, wsType_Container, WSTYPE_PLACEHOLDER},
		category			= CAT_ROCKETS,
		connNamePattern		= "Tube_%02d",
		Count				= 25,
		Cx_pil				= 0.00059912109375,
		displayNamePrfx		= ("LR-25 - 25 x %s, 50 %s"):format(LocStr.UnGdRkts, LocStr.mm),
		modelFileName		= "LR-25",
		Picture				= "LR25.png",
		Weight_Empty		= 46,
		weapon_container_data = {
			nurs_args			= { 4 },
			launched_arg_state	= 1,
		}
	},
	M260 = {
		attribute			= {wsType_Weapon, wsType_NURS, wsType_Container, WSTYPE_PLACEHOLDER},
		category			= CAT_ROCKETS,
		connNamePattern		= "tube_%02d",
		Count				= 7,
		Cx_pil				= 0.001708984375,
		displayNamePrfx		= ("M260 - 7 x %s, 70 %s"):format(LocStr.UnGdRkts, LocStr.mm),
		Elements			= ContainerWithSubshapes("OH58D_M260", "hydra_m151he", 7, "tube_%02d"),
		modelFileName		= "OH58D_M260",
		Picture				= "LAU68.png",
		Weight_Empty		= 15.9,
		genericWpnName		= "Hydra 70",
		zoneNumbersTable	= {
			["A"] = { 4, 5, 6, 7 },
			["B"] = { 1, 2, 3 }
		},
	},
	M261 = {
		attribute			= {wsType_Weapon, wsType_NURS, wsType_Container, WSTYPE_PLACEHOLDER},
		category			= CAT_ROCKETS,
		connNamePattern		= "tube_%d",
		Count				= 19,
		Cx_pil				= 0.001708984375,
		displayNamePrfx		= ("M261 - 19 x %s, 70 %s"):format(LocStr.UnGdRkts, LocStr.mm),
		Elements			= ContainerWithSubshapes("M261", "hydra_m151he", 19, "tube_%d"),
		modelFileName		= "M261",
		Picture				= "LAU61.png",
		Weight_Empty		= 39.4, -- per SME feedback
		genericWpnName		= "Hydra 70",
		zoneNumbersTable	= {
			["A"] = { 1, 2, 3, 4, 7, 8, 12, 13, 16, 17, 18, 19 },
			["B"] = { 5, 6, 14, 15 },
			["E"] = { 9, 10, 11 }
		},
		zoneRemapper		= function (zoneName) if zoneName == "C" then zoneName = "A" elseif zoneName == "D" then zoneName = "B" end return zoneName end
	},
	MATRAF1 = {
		attribute			= {wsType_Weapon, wsType_NURS, wsType_Container, WSTYPE_PLACEHOLDER},
		category			= CAT_ROCKETS,
		Count				= 36,
		Cx_pil				= 0.016 / 25,
		displayNamePrfx		= ("MATRA F1 - 36 x %s, 68 %s"):format(LocStr.UnGdRkts, LocStr.mm),
		Elements			= ContainerWithSubshapes("Matra-F1-Rocket", "SA342_SNEB68", 36),
		modelFileName		= "Matra-F1-Rocket",
		Picture				= "LAU61.png",
		Weight_Empty		= 88
	},
	MATRAF4 = {
		attribute			= {wsType_Weapon, wsType_NURS, wsType_Container, WSTYPE_PLACEHOLDER},
		category			= CAT_ROCKETS,
		Count				= 18,
		Cx_pil				= 0.007 / 25,
		displayNamePrfx		= ("MATRA F4 - 18 x %s, 68 %s"):format(LocStr.UnGdRkts, LocStr.mm),
		Elements			= {{ ShapeName = "M-2000C_LRF4", IsAdapter = true }},
		modelFileName		= "M-2000C_LRF4",
		Picture				= "LAU61.png",
		Weight_Empty		= 96.5
	},
	TELSON8 = {
		attribute			= {wsType_Weapon, wsType_NURS, wsType_Container, WSTYPE_PLACEHOLDER},
		category			= CAT_ROCKETS,
		Count				= 8,
		Cx_pil				= 0.007,
		displayNamePrfx		= ("Telson 8 - 8 x %s, 68 %s"):format(LocStr.UnGdRkts, LocStr.mm),
		Elements			= ContainerWithSubshapes("SA342_Telson8", "SA342_SNEB68", 8),
		modelFileName		= "SA342_Telson8",
		Picture				= "SNEB68.png",
		Weight_Empty		= 30
	},
}

local function RocketContainer(shape)
	return {{ ShapeName = shape, IsAdapter = true }}
end

---A function to declare a lot of rockets (and create their descriptors) w/o copypasta; indices in the paramTable are unique weapon names.
---@param paramTable	table	The table of parameters of the rockets (what FM/engine/warhead is to be used; some parameter overrides, etc.);
--- basically, the only required parameter is schemeName, but you'll want to use some more to achieve your goals.
---@return				table	OverallResult	Total descriptor list for the declared rockets (for later use)
function BatchDeclareRockets(paramTable)
	OverallResult = {}

	for index, rktParams in pairs(paramTable.variantSpecificParams) do
		currentResult = {}

		local rktParamOverrides = rktParams.paramOverrides or {}

		-- Parameters for most types of rockets (those without "launcher" block), excluding dispenser/illumination rockets
		local warheadParams		= rktParams.payloadParams.warhead or {}
		-- Parameters for dispenser/illumination rockets (those with "launcher" block)
		local launcherParams	= rktParams.payloadParams.launcher or {}

		-- Engine mass w/propellant
		local engineTotalMass	= paramTable.commonParams.baseEngine.total_mass or 0
		-- Engine length
		local engineLength		= paramTable.commonParams.baseEngine.length or 0
		-- Payload (warhead) full mass
		local payloadMass		= warheadParams.mass
			or launcherParams.ammunition and launcherParams.ammunition.client and launcherParams.ammunition.client.fm.mass
			or launcherParams.ammunition and launcherParams.ammunition.fm.mass
			or 0
		-- Payload (warhead) length
		local payloadLength		= warheadParams.length
			or launcherParams.ammunition and launcherParams.ammunition.client and launcherParams.ammunition.client.fm.L
			or launcherParams.ammunition and launcherParams.ammunition.fm.L
			or 0
		-- Final displayName
		local displayNameFinal	= rktParams.displayName

		-- If we need to add a prefix
		if paramTable.commonParams.displayNamePrfx then
			displayNameFinal = paramTable.commonParams.displayNamePrfx .. " " .. displayNameFinal
		end

		-- Create the descriptor table
		currentResult = {
			category			= CAT_ROCKETS,
			name				= index,
			user_name			= rktParams.displayName,
			displayName			= displayNameFinal,
			display_name_short	= rktParams.displayNameShort,
			scheme				= rktParams.schemeName,
			class_name			= "wAmmunitionNURS",
			model				= rktParams.modelFileName or paramTable.commonParams.modelFileName,
			wsTypeOfWeapon		= rktParamOverrides.wsTypeOfWeapon or {wsType_Weapon, wsType_NURS, wsType_Rocket, rktParamOverrides.wsTypeOfWeaponL4 or WSTYPE_PLACEHOLDER},

			engine				= paramTable.commonParams.baseEngine,

			fm					= specialize_fm_parameters(
				paramTable.commonParams.baseFM,
				{
					mass		= engineTotalMass + payloadMass,
					L			= engineLength + payloadLength,
					I			= 1 / 12 * (engineTotalMass + payloadMass) * (engineLength + payloadLength) ^ 2,
					shapeName	= rktParams.modelFileName,
					freq		= paramTable.commonParams.baseEngine.induced_RPS or 7
				}
			),

			properties = rktParams.properties or paramTable.commonParams.properties or {
				dist_min	= 500,
				dist_max	= 4000
			},

			shape_table_data = {{
				name		= index,
				file		= rktParams.modelFileName or paramTable.commonParams.modelFileName,
				username	= displayNameFinal,
				index		= rktParamOverrides.wsTypeOfWeaponL4 or WSTYPE_PLACEHOLDER
			}}
		}

		if warheadParams.mass then
			currentResult.warhead = warheadParams
		else
			currentResult.launcher = { ammunition_name = launcherParams.ammunition and launcherParams.ammunition.name or launcherParams.ammunition_name }
			currentResult.control = { delay = launcherParams.delay }
		end

		OverallResult[index] = currentResult
		declare_weapon(OverallResult[index])
	end

	return OverallResult
end

rockets_data =
{
	["LAU-68_M151"]		= {name = "LAU-68 pod - 7 x 2.75\" Hydra, UnGd Rkts M151, HE", 			payload_CLSID = "{A021F29D-18AB-4d3e-985C-FC9C60E35E9E}", mass = 113.9, mass_empty = 40.4, wsType = {4, 7, 33, 147}, Cx = 0.00146484375, picture = "LAU68.png"},
	["LAU-68_MK5"]		= {name = "LAU-68 pod - 7 x 2.75\" Hydra, UnGd Rkts Mk5, HEAT",			payload_CLSID = "{174C6E6D-0C3D-42ff-BCB3-0853CB371F5C}", mass = 102.1, mass_empty = 40.4, wsType = {4, 7, 33, 145}, Cx = 0.00146484375, picture = "LAU68.png"},
	["LAU-68_M156"]		= {name = "LAU-68 pod - 7 x 2.75\" Hydra, UnGd Rkts M156, Wht Phos", 	payload_CLSID = "{4F977A2A-CD25-44df-90EF-164BFA2AE72F}", mass = 114.5, mass_empty = 40.4, wsType = {4,	7, 33, 148}, Cx = 0.00146484375, picture = "LAU68.png"},
	["LAU-68_MK61"]		= {name = "LAU-68 pod - 7 x 2.75\" Hydra, UnGd Rkts Mk61, Practice", 	payload_CLSID = "{65396399-9F5C-4ec3-A7D2-5A8F4C1D90C4}", mass = 104.2, mass_empty = 40.4, wsType = {4,	7, 33, 146}, Cx = 0.00146484375, picture = "LAU68.png"},
	["LAU-68_WTU-1/B"]	= {name = "LAU-68 pod - 7 x 2.75\" Hydra, UnGd Rkts WTU-1/B, Practice", payload_CLSID = "{1F7136CB-8120-4e77-B97B-945FF01FB67C}", mass = 113.2, mass_empty = 40.4, wsType = {4,	7, 33, 149}, Cx = 0.00146484375, picture = "LAU68.png"},
	["LAU-68_MK1"]		= {name = "LAU-68 pod - 7 x 2.75\" Hydra, UnGd Rkts Mk1, Practice",		payload_CLSID = "{FC85D2ED-501A-48ce-9863-49D468DDD5FC}", mass = 104.2, mass_empty = 40.4, wsType = {4, 7, 33, 144}, Cx = 0.00146484375, picture = "LAU68.png"},
	["LAU-68_M257"]		= {name = "LAU-68 pod - 7 x 2.75\" Hydra, UnGd Rkts M257, Para Illum",	payload_CLSID = "{647C5F26-BDD1-41e6-A371-8DE1E4CC0E94}", mass = 118.8, mass_empty = 40.4, wsType = {4, 7, 33, 151}, Cx = 0.00146484375, picture = "LAU68.png"},
	["LAU-68_M274"]		= {name = "LAU-68 pod - 7 x 2.75\" Hydra, UnGd Rkts M274, Practice Smk",payload_CLSID = "{0877B74B-5A00-4e61-BA8A-A56450BA9E27}", mass = 113.2, mass_empty = 40.4, wsType = {4, 7, 33, 150}, Cx = 0.00146484375, picture = "LAU68.png"},
	
	["LAU-131_M151"]	= {name = "LAU-131 pod - 7 x 2.75\" Hydra, UnGd Rkts M151, HE", 		payload_CLSID = "{69926055-0DA8-4530-9F2F-C86B157EA9F6}", mass = 102.3, mass_empty = 29.5, wsType = {4,	7, 33, 147}, Cx = 0.00146484375, picture = "LAU131.png"},
	["LAU-131_MK5"]		= {name = "LAU-131 pod - 7 x 2.75\" Hydra, UnGd Rkts Mk5, HEAT", 		payload_CLSID = "{319293F2-392C-4617-8315-7C88C22AF7C4}", mass = 91.2, 	mass_empty = 29.5, wsType = {4,	7, 33, 145}, Cx = 0.00146484375, picture = "LAU131.png"},
	["LAU-131_M156"]	= {name = "LAU-131 pod - 7 x 2.75\" Hydra, UnGd Rkts M156, Wht Phos", 	payload_CLSID = "{2AF2EC3F-9065-4de5-93E1-1739C9A71EF7}", mass = 103.6, mass_empty = 29.5, wsType = {4,	7, 33, 148}, Cx = 0.00146484375, picture = "LAU131.png"},
	["LAU-131_MK61"]	= {name = "LAU-131 pod - 7 x 2.75\" Hydra, UnGd Rkts Mk61, Practice", 	payload_CLSID = "{1CA5E00B-D545-4ff9-9B53-5970E292F14D}", mass = 93.3, 	mass_empty = 29.5, wsType = {4,	7, 33, 146}, Cx = 0.00146484375, picture = "LAU131.png"},
	["LAU-131_WTU-1/B"]	= {name = "LAU-131 pod - 7 x 2.75\" Hydra, UnGd Rkts WTU-1/B, Practice",payload_CLSID = "{DDCE7D70-5313-4181-8977-F11018681662}", mass = 102.3, mass_empty = 29.5, wsType = {4,	7, 33, 149}, Cx = 0.00146484375, picture = "LAU131.png"},
	["LAU-131_MK1"]		= {name = "LAU-131 pod - 7 x 2.75\" Hydra, UnGd Rkts Mk1, Practice",	payload_CLSID = "{D22C2D63-E5C9-4247-94FB-5E8F3DE22B71}", mass = 93.3, 	mass_empty = 29.5, wsType = {4, 7, 33, 144}, Cx = 0.00146484375, picture = "LAU131.png"},
	["LAU-131_M257"]	= {name = "LAU-131 pod - 7 x 2.75\" Hydra, UnGd Rkts M257, Para Illum",	payload_CLSID = "{DAD45FE5-CFF0-4a2b-99D4-5D044D3BC22F}", mass = 107.9, mass_empty = 29.5, wsType = {4, 7, 33, 151}, Cx = 0.00146484375, picture = "LAU131.png"},
	["LAU-131_M274"]	= {name = "LAU-131 pod - 7 x 2.75\" Hydra, UnGd Rkts M274, Practice Smk",payload_CLSID = "{6D6D5C07-2A90-4a68-9A74-C5D0CFFB05D9}",mass = 102.3,	mass_empty = 29.5, wsType = {4, 7, 33, 150}, Cx = 0.00146484375, picture = "LAU131.png"},
	
	["LAU-10"]			= {name = "LAU-10 pod - 4 x 127mm ZUNI, UnGd Rkts Mk71, HE/FRAG", 		payload_CLSID = "{F3EFE0AB-E91A-42D8-9CA2-B63C91ED570A}", mass = 288.9, mass_empty = 61.7, wsType = {4, 7, 33, 37},  Cx = 0.001708984375, picture = "LAU10.png"},
	["LAU-61"]			= {name = "LAU-61 pod - 19 x 2.75\" Hydra, UnGd Rkts M151, HE",			payload_CLSID = "{FD90A1DC-9147-49FA-BF56-CB83EF0BD32B}", mass = 273.4, mass_empty = 73.9, wsType = {4, 7, 33, 147}, Cx = 0.001708984375, picture = "LAU61.png"},
}

declare_loadout({
				category		=	CAT_ROCKETS,
				CLSID			=	"{LAU_61R}",
				Picture			=	"LAU61.png",
				Cx_pil			=	0.001708984375,
				displayName		=	_("LAU-61R pod - 19 x 2.75\" Hydra, UnGd Rkts M151, HE"),
				Count			=	19,
				Elements		=	RocketContainer("LAU-61"),
				Weight			=	73.9+19*10.4,
				wsTypeOfWeapon	=	"weapons.nurs.HYDRA_70_M151",
				attribute		=	{4,	7,	32,	9},
})

declare_loadout({
				category		= CAT_ROCKETS,
				CLSID			= "{LAU_10R}",
				Picture			= "LAU10.png",
				Cx_pil			= 0.001708984375,
				displayName		= _("LAU-10R pod - 4 x 127mm ZUNI, UnGd Rkts Mk71, HE/FRAG"),
				Count			= 4,
				Weight			= 61.7 + 4 * 56.8,
				Weight_Empty	= 61.7,
				wsTypeOfWeapon	= {4, 7, 33, 37},
				attribute		= {4, 7, 32, 8},
				Elements		= {
					{ ShapeName = "LAU-10" , IsAdapter = true},
					--rockets itself
					{ ShapeName = "zuni", Position = {0.1, 0.049, 0.072} },
					{ ShapeName = "zuni", Position = {0.1, 0.049, -0.073} },
					{ ShapeName = "zuni", Position = {0.1, -0.095, 0.072} },
					{ ShapeName = "zuni", Position = {0.1, -0.095, -0.073} },
				},
})

local function bru_33_2xlau(clsid,element,left,right)
	local lau_variant = rockets_data[element] or rockets_data["LAU-68_M151"]
	local ret = {
		category			=	CAT_ROCKETS,
		CLSID				=	clsid,
		Picture				=	lau_variant.picture,
		wsTypeOfWeapon		=	lau_variant.wsType,
		attribute			=	{4,	7,	32,	WSTYPE_PLACEHOLDER},
		Cx_pil				=	0.00244140625,
		Elements 			= {{ShapeName	=	"BRU_33A",IsAdapter  	   =   true  },}
	}
	if left then 	ret.Elements[#ret.Elements + 1] = {payload_CLSID = lau_variant.payload_CLSID, connector_name =	"Point02"}	end
	if right then	ret.Elements[#ret.Elements + 1] = {payload_CLSID = lau_variant.payload_CLSID, connector_name =	"Point01"}	end

	local sz = #ret.Elements - 1
	ret.Count  = sz
	ret.Weight_Empty = bru_33VER_mass + sz * lau_variant.mass_empty
	ret.Weight = bru_33VER_mass + sz * lau_variant.mass

	ret.Cx_pil = ret.Cx_pil + sz * lau_variant.Cx

	if sz > 1 then
		ret.displayName =	_("BRU-33").." with 2 x "..lau_variant.name
	else
		ret.displayName =	_("BRU-33").." with 1 x "..lau_variant.name
	end
	declare_loadout(ret)
end

---Fills the Elements table according to the launcher's zoning
---@param launcherData table Info about the launcher (see RLData)
---@param loadoutTable table Desired loadout for each zone
---@return table
local function fillZonedLauncher(launcherData, loadoutTable)
	local result = {}
	local tubeCLSIDPairs = {}

	result[#result + 1] = {
		ShapeName = launcherData.modelFileName,
		IsAdapter = true
	}

	for zoneName, CLSID in pairs(loadoutTable) do
		if launcherData.zoneRemapper then
			zoneName = launcherData.zoneRemapper(zoneName)
		end

		-- make CLSIDs correspond to tube numbers
		for __, tubeNumber in ipairs(launcherData.zoneNumbersTable[zoneName]) do
			tubeCLSIDPairs[tubeNumber] = CLSID
		end
	end

	for i, CLSID in ipairs(tubeCLSIDPairs) do
		result[i + 1] = {
			payload_CLSID = CLSID,
			connector_name = string.format(launcherData.connNamePattern, i)
		}
	end

	return result
end

bru_33_2xlau("{BRU33_2*LAU68}"	,	"LAU-68_M151",true,true)
bru_33_2xlau("{BRU33_LAU68}"	,	"LAU-68_M151",true, false)
bru_33_2xlau("{BRU33_LAU68_MK5}",	"LAU-68_MK5",true,false)
bru_33_2xlau("{BRU33_2*LAU68_MK5}",	"LAU-68_MK5",true,true)
bru_33_2xlau("{BRU33_LAU10}",		"LAU-10",true,false)
bru_33_2xlau("{BRU33_2*LAU10}",		"LAU-10",true,true)
-- bru_33_2xlau("{BRU33_LAU61}",		"LAU-61",true,false)
bru_33_2xlau("{BRU33_2*LAU61}",		"LAU-61",true,true)

-- For LAU-68/131
local function bru_42_ls_2x_lau_68_131(clsid,element,isleft)
	local lau_variant = rockets_data[element] or rockets_data["LAU-68_M151"]
	local ret = {
		category			=	CAT_ROCKETS,
		CLSID				=	clsid,
		Picture				=	lau_variant.picture,
		wsTypeOfWeapon		=	lau_variant.wsType,
		attribute			=	{4,	7,	32,	WSTYPE_PLACEHOLDER},
		Cx_pil				=	0.0005,
		Elements 			= 	{}
	}
	
	ret.Elements[#ret.Elements + 1] = { ShapeName	= "BRU-42_LS", IsAdapter  =   true}
	
	if isleft then	
		ret.Elements[#ret.Elements + 1] = {payload_CLSID = lau_variant.payload_CLSID, connector_name =	"Point03"}
	else
		ret.Elements[#ret.Elements + 1] = {payload_CLSID = lau_variant.payload_CLSID, connector_name =	"Point02"}	
	end
	ret.Elements[#ret.Elements + 1] = {payload_CLSID = lau_variant.payload_CLSID, connector_name =	"Point01"}

	local sz = #ret.Elements - 1
	ret.Count  = sz
	ret.Weight_Empty = 56.25 + sz * lau_variant.mass_empty
	ret.Weight = 56.25 + sz * lau_variant.mass
	ret.Cx_pil = ret.Cx_pil + sz * lau_variant.Cx
	ret.displayName =	_("BRU-42 with "..sz.." x "..lau_variant.name)
	
	declare_loadout(ret)
end

bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU68_HYDRA_70_M151_L}", 	"LAU-68_M151", true)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU68_HYDRA_70_M151_R}", 	"LAU-68_M151", false)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU68_HYDRA_70_MK5_L}", 	"LAU-68_MK5", true)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU68_HYDRA_70_MK5_R}", 	"LAU-68_MK5", false)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU68_HYDRA_70_M156_L}", 	"LAU-68_M156", true)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU68_HYDRA_70_M156_R}", 	"LAU-68_M156", false)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU68_HYDRA_70_MK61_L}", 	"LAU-68_MK61", true)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU68_HYDRA_70_MK61_R}", 	"LAU-68_MK61", false)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU68_HYDRA_70_WTU1B_L}", 	"LAU-68_WTU-1/B", true)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU68_HYDRA_70_WTU1B_R}", 	"LAU-68_WTU-1/B", false)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU68_HYDRA_70_MK1_L}", 	"LAU-68_MK1", true)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU68_HYDRA_70_MK1_R}", 	"LAU-68_MK1", false)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU68_HYDRA_70_M257_L}", 	"LAU-68_M257", true)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU68_HYDRA_70_M257_R}", 	"LAU-68_M257", false)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU68_HYDRA_70_M274_L}", 	"LAU-68_M274", true)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU68_HYDRA_70_M274_R}", 	"LAU-68_M274", false)

bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU131_HYDRA_70_M151_L}", 	"LAU-131_M151", true)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU131_HYDRA_70_M151_R}", 	"LAU-131_M151", false)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU131_HYDRA_70_MK5_L}", 	"LAU-131_MK5", true)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU131_HYDRA_70_MK5_R}", 	"LAU-131_MK5", false)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU131_HYDRA_70_M156_L}", 	"LAU-131_M156", true)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU131_HYDRA_70_M156_R}", 	"LAU-131_M156", false)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU131_HYDRA_70_MK61_L}", 	"LAU-131_MK61", true)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU131_HYDRA_70_MK61_R}", 	"LAU-131_MK61", false)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU131_HYDRA_70_WTU1B_L}", 	"LAU-131_WTU-1/B", true)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU131_HYDRA_70_WTU1B_R}", 	"LAU-131_WTU-1/B", false)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU131_HYDRA_70_MK1_L}", 	"LAU-131_MK1", true)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU131_HYDRA_70_MK1_R}", 	"LAU-131_MK1", false)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU131_HYDRA_70_M257_L}", 	"LAU-131_M257", true)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU131_HYDRA_70_M257_R}", 	"LAU-131_M257", false)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU131_HYDRA_70_M274_L}", 	"LAU-131_M274", true)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU131_HYDRA_70_M274_R}", 	"LAU-131_M274", false)



-- TER-9/A with 2 rockets
-- Because on one and on three modules have already been created before that
local function ter_9a_2rockets(element_)
	local rocket_variant = rockets_data[element_] or rockets_data["LAU-68_M151"]

	function createTableTER9(isLeft)
		local side_
		local secondConnector

		if isLeft then
			side_ = "L"
			secondConnector = "Point02"
		else
			side_ = "R"
			secondConnector = "Point03"
		end
		
		return	{
			category		=	CAT_ROCKETS,
			CLSID			=	"{TER_9A_2"..side_.."*"..element_.."}",
			Picture			=	rocket_variant.picture,
			wsTypeOfWeapon	=	rocket_variant.wsType,
			displayName		=  _("TER-9A with 2 x ")..rocket_variant.name,
			attribute		=	{wsType_Weapon, wsType_NURS, wsType_Container, WSTYPE_PLACEHOLDER},
			Cx_pil			=	0.00045 + 2 * rocket_variant.Cx,
			Cx_item			= 	-0.00001,
			Count			=	14,
			Weight			=	ter_9a_mass + 2 * rocket_variant.mass,
			Elements		= 	{{ ShapeName = "TER-9A", IsAdapter = true },
								{ShapeName	= element_,	payload_CLSID = rocket_variant.payload_CLSID, connector_name =	"Point01"},
								{ShapeName	= element_,	payload_CLSID = rocket_variant.payload_CLSID, connector_name =	secondConnector}},

		}
	end

	declare_loadout(createTableTER9(true))
	declare_loadout(createTableTER9(false))
end

ter_9a_2rockets("LAU-68_MK5")		-- {TER_9A_2L*LAU-68_MK5} && {TER_9A_2R*LAU-68_MK5}
ter_9a_2rockets("LAU-68_M151")		-- {TER_9A_2L*LAU-68_M151} && {TER_9A_2R*LAU-68_M151}
ter_9a_2rockets("LAU-68_M156")		-- {TER_9A_2L*LAU-68_M156} && {TER_9A_2R*LAU-68_M156}
ter_9a_2rockets("LAU-68_MK61")		-- {TER_9A_2L*LAU-68_MK61} && {TER_9A_2R*LAU-68_MK61}
ter_9a_2rockets("LAU-68_WTU-1/B")	-- {TER_9A_2L*LAU-68_WTU-1/B} && {TER_9A_2R*LAU-68_WTU-1/B}

ter_9a_2rockets("LAU-131_MK5")		-- {TER_9A_2L*LAU-131_MK5} && {TER_9A_2R*LAU-131_MK5}
ter_9a_2rockets("LAU-131_M151")		-- {TER_9A_2L*LAU-131_M151} && {TER_9A_2R*LAU-131_M151}
ter_9a_2rockets("LAU-131_M156")		-- {TER_9A_2L*LAU-131_M156} && {TER_9A_2R*LAU-131_M156}
ter_9a_2rockets("LAU-131_MK61")		-- {TER_9A_2L*LAU-131_MK61} && {TER_9A_2R*LAU-131_MK61}
ter_9a_2rockets("LAU-131_WTU-1/B")	-- {TER_9A_2L*LAU-131_WTU-1/B} && {TER_9A_2R*LAU-131_WTU-1/B}

declare_loadout({
	category		 = CAT_ROCKETS,
	CLSID			 = "{UB-16-57UMP}",
	displayName		 =	_("UB-16UM pod - 16 x S-5KO, 57mm UnGd Rkts, HEAT/Frag"),
	wsTypeOfWeapon	 =	{4,	7,	33,	31},
	attribute	 	 =	{4,	7,	32,	WSTYPE_PLACEHOLDER},
	Count			 =	16,
	Picture			 = "ub16.png",
	Weight_Empty	 = 57,
	Weight			 = 138,
	Cx_pil			 =	0.00196533203125,
	Elements		 = ContainerWithSubshapes("UB-16-57UMP", "S-5KO", 16, "tube_%d")
})

declare_loadout({
	category		=	CAT_ROCKETS,
	CLSID			=	"{FC56DF80-9B09-44C5-8976-DCFAFF219062}",
	Picture			=	"rus_B-13L_S-13OF.png",
	PictureBlendColor = "0xffffffff",
	Cx_pil			=	0.00159912109375,
	displayName		=	_("B-13L pod - 5 x S-13-OF, 122mm UnGd Rkts, Blast/Frag"),
	Count			=	5,
	Elements		=	ContainerWithSubshapes("UB-13", "C-13", 5, "tube_%d"),
	Weight			=	510,
	wsTypeOfWeapon	=	{4,	7,	33,	33},
	attribute		=	{4,	7,	32,	1},
})

declare_loadout({
	category 		= CAT_ROCKETS,
	CLSID	 		= "{TWIN_B13L_5OF}",
	wsTypeOfWeapon	= {4,	7,	33,	33},
	attribute		= {4,	7,	32,	WSTYPE_PLACEHOLDER},
	Count 			= 10,
	Cx_pil			= 0.0004,
	Picture			= "B13.png",
	displayName		=	_("2 x B-13L pods - 10 x S-13-OF, 122mm UnGd Rkts, Blast/Frag"),
	Weight			= 32 + 2*(160+5*69),
	JettisonSubmunitionOnly = true,
	Elements		=	{
	   {ShapeName = "su-27-twinpylon",IsAdapter = true},
	   {payload_CLSID = "{FC56DF80-9B09-44C5-8976-DCFAFF219062}", connector_name = "S-25-L"},
	   {payload_CLSID = "{FC56DF80-9B09-44C5-8976-DCFAFF219062}", connector_name = "S-25-R"},
	}
})



declare_loadout({
	category 		= CAT_ROCKETS,
	CLSID	 		= "{APU_68_S-24}",
	wsTypeOfWeapon	= {4,	7,	33,	34},
	attribute		= {4,	7,	32,	WSTYPE_PLACEHOLDER},
	Count 			= 1,
	Cx_pil			= 0.0002,
	Picture			= "RBK250.png",
	displayName		= _("APU-68 - S-24B"),
	Weight			= 38.5 + 235,
	Elements		=	{
	   {ShapeName = "apu-68um3",IsAdapter = true},
	   {payload_CLSID = "{1FA14DEA-8CDB-45AD-88A8-EC068DF1E65A}", connector_name = "Point Pilon s24"},
	}
})

local AGR_20_M151_UNGUIDED =
{
	category		= CAT_ROCKETS,
	name			= "AGR_20_M151_unguided",
	user_name		= _("APKWS M151 unguided"),
	display_name_short 	= "M151",
	scheme			= "nurs-standard",
	class_name		= "wAmmunitionNURS",
	model			= "AGR_20",
	wsTypeOfWeapon 	= {wsType_Weapon, wsType_NURS, wsType_Rocket, WSTYPE_PLACEHOLDER},

	warhead		= predefined_warhead("HYDRA_70_M151"),

	shape_table_data =
	{
		{
			name  = "AGR_20_M151_unguided";
			file  = "AGR_20";
			life  = 1;
			fire  = { 0, 1};
			username = _("APKWS M151 unguided"),
			index 	 = WSTYPE_PLACEHOLDER,
		},
	},

	engine = {
		fuel_mass   				= 7.5,
		impulse     				= 210,
		boost_time  				= 0,
		work_time   				= 1.8,
		boost_factor				= 1,
		nozzle_position 	   		= {{-0.95, 0, 0}},
		nozzle_orientationXYZ		= {{0, 0, 0}},
		tail_width  				= 0.052,
		boost_tail  				= 1,
		work_tail   				= 1,
		smoke_color					= {0.5, 0.474, 0.443},
		smoke_transparency			= 0.5,
	},

	fm = {
		mass        = 15,
		caliber     = 0.07,
		cx_coeff    = {1, 1.5, 0.68, 0.7, 1.75},
		L           = 1.81,
		I           = 1 / 12 * 15 * 1.81 * 1.81,
		Ma          = 0.84,
		Mw          = 3.5,
		shapeName   = "AGR_20",
		wind_sigma	= 5,
		wind_time	= 1.8,
	},

	properties = {
		dist_min = 500,
		dist_max = 4000,
	},
}

declare_weapon(AGR_20_M151_UNGUIDED)

local AGR_20_M282_UNGUIDED =
{
	category		= CAT_ROCKETS,
	name			= "AGR_20_M282_unguided",
	user_name		= _("APKWS M282 unguided"),
	display_name_short 	= "M-282",
	scheme			= "nurs-standard",
	class_name		= "wAmmunitionNURS",
	model			= "AGR_20",
	wsTypeOfWeapon 	= {wsType_Weapon, wsType_NURS, wsType_Rocket, WSTYPE_PLACEHOLDER},

	warhead		= predefined_warhead("HYDRA_70_M282"),

	shape_table_data =
	{
		{
			name  = "AGR_20_M282_unguided";
			file  = "AGR_20";
			life  = 1;
			fire  = { 0, 1};
			username = _("APKWS M282 unguided"),
			index 	 = WSTYPE_PLACEHOLDER,
		},
	},

	engine = {
		fuel_mass   				= 7.5,
		impulse     				= 210,
		boost_time  				= 0,
		work_time   				= 1.8,
		boost_factor				= 1,
		nozzle_position 	   		= {{-0.95, 0, 0}},
		nozzle_orientationXYZ		= {{0, 0, 0}},
		tail_width  				= 0.052,
		boost_tail  				= 1,
		work_tail   				= 1,
		smoke_color					= {0.5, 0.474, 0.443},
		smoke_transparency			= 0.5,
	},

	fm = {
		mass        = 17,
		caliber     = 0.07,
		cx_coeff    = {1, 1.5, 0.68, 0.7, 1.75},
		L           = 1.81,
		I           = 1 / 12 * 15 * 1.81 * 1.81,
		Ma          = 0.84,
		Mw          = 3.5,
		shapeName   = "AGR_20",
		wind_sigma	= 5,
		wind_time	= 1.8,
	},

	properties = {
		dist_min = 500,
		dist_max = 4000,
	},
}

declare_weapon(AGR_20_M282_UNGUIDED)

local Hydra_Mk66_Variants = BatchDeclareRockets({
	commonParams = {
		baseFM			= predefined_fm("Hydra70"),
		baseEngine		= predefined_engine("Mk66"),
	},
	variantSpecificParams = {
		["HYDRA_70_M151"] = {
			displayName			= _("Hydra 70 M151 HE"),
			displayNameShort	= "M151",
			schemeName			= "nurs-standard",
			modelFileName		= "hydra_m151he",
			payloadParams		= { warhead = predefined_warhead("HYDRA_70_M151") },
			paramOverrides		= { wsTypeOfWeaponL4 = HYDRA_70_M151 }
		},
		["HYDRA_70_M151_M433"] = {
			displayName			= _("Hydra 70 M151 HE, M433 RC Fuze"),
			displayNameShort	= "M151",
			schemeName			= "nurs-standard",
			modelFileName		= "hydra_m433",
			payloadParams		= { warhead = predefined_warhead("HYDRA_70_M151") },
		},
		["HYDRA_70_M156"] = {
			displayName			= _("Hydra 70 M156 SM"),
			displayNameShort	= "M156",
			schemeName			= "nurs-marker",
			modelFileName		= "hydra_m156",
			payloadParams		= { warhead = predefined_warhead("HYDRA_70_M156") },
			paramOverrides		= { wsTypeOfWeaponL4 = HYDRA_70_M156 }
		},
		["HYDRA_70_M229"] = {
			displayName			= _("Hydra 70 M229 HE"),
			displayNameShort	= "M229",
			schemeName			= "nurs-standard",
			modelFileName		= "hydra_m229",
			payloadParams		= { warhead = predefined_warhead("HYDRA_70_M229") },
		},
		["HYDRA_70_M257"] = {
			displayName			= _("Hydra 70 M257 IL"),
			displayNameShort	= "M257",
			schemeName			= "nurs-light",
			modelFileName		= "Hydra_M257",
			payloadParams		= { launcher = { ammunition = M257_FLARE, delay = 10.1 } },
			properties = {
				dist_min	= 1000,
				dist_max	= 4000,
				max_TOF		= 10.5 -- exaggerated a bit to avoid AI overshooting the target due to non-instant deceleration of the flare element
			},
			paramOverrides	= { wsTypeOfWeaponL4 = HYDRA_70_M257 }
		},
		["HYDRA_70_M259"] = {
			displayName			= _("Hydra 70 M259 SM"),
			displayNameShort	= "M259",
			schemeName			= "nurs-marker",
			modelFileName		= "hydra_wdu4aa", -- best match available rn
			payloadParams		= { warhead = predefined_warhead("HYDRA_70_M259") },
		},
		["HYDRA_70_M274"] = {
			displayName			= _("Hydra 70 M274 TP-SM"),
			displayNameShort	= "M274",
			schemeName			= "nurs-marker",
			modelFileName		= "hydra_m156", -- best match available rn
			payloadParams		= { warhead = predefined_warhead("HYDRA_70_M274") },
			paramOverrides		= { wsTypeOfWeaponL4 = HYDRA_70_M274 }
		},
		["HYDRA_70_M282"] = {
			displayName			= _("Hydra 70 M282 MPP"),
			displayNameShort	= "M282",
			schemeName			= "nurs-standard",
			modelFileName		= "hydra_m282",
			payloadParams		= { warhead = predefined_warhead("HYDRA_70_M282") },
		},
	}
})

declare_loadout({
	category		=	CAT_ROCKETS,
	CLSID			=	"{M151_HYDRA}",
	Count			=	1,
	Elements		=	{{ ShapeName = Hydra_Mk66_Variants["HYDRA_70_M151"].model }},
	Weight			=	Hydra_Mk66_Variants["HYDRA_70_M151"].fm.mass,
	wsTypeOfWeapon	=	Hydra_Mk66_Variants["HYDRA_70_M151"].wsTypeOfWeapon,
	attribute		=	Hydra_Mk66_Variants["HYDRA_70_M151"].wsTypeOfWeapon,
})

declare_loadout({
	category		=	CAT_ROCKETS,
	CLSID			=	"{M156_HYDRA}",
	Count			=	1,
	Elements		=	{{ ShapeName = Hydra_Mk66_Variants["HYDRA_70_M156"].model }},
	Weight			=	Hydra_Mk66_Variants["HYDRA_70_M156"].fm.mass,
	wsTypeOfWeapon	=	Hydra_Mk66_Variants["HYDRA_70_M156"].wsTypeOfWeapon,
	attribute		=	Hydra_Mk66_Variants["HYDRA_70_M156"].wsTypeOfWeapon,
})

declare_loadout({
	category		=	CAT_ROCKETS,
	CLSID			=	"{M229_HYDRA}",
	Count			=	1,
	Elements		=	{{ ShapeName = Hydra_Mk66_Variants["HYDRA_70_M229"].model }},
	Weight			=	Hydra_Mk66_Variants["HYDRA_70_M229"].fm.mass,
	wsTypeOfWeapon	=	Hydra_Mk66_Variants["HYDRA_70_M229"].wsTypeOfWeapon,
	attribute		=	Hydra_Mk66_Variants["HYDRA_70_M229"].wsTypeOfWeapon,
})

declare_loadout({
	category		=	CAT_ROCKETS,
	CLSID			=	"{M257_HYDRA}",
	Count			=	1,
	Elements		=	{{ ShapeName = Hydra_Mk66_Variants["HYDRA_70_M257"].model }},
	Weight			=	Hydra_Mk66_Variants["HYDRA_70_M257"].fm.mass,
	wsTypeOfWeapon	=	Hydra_Mk66_Variants["HYDRA_70_M257"].wsTypeOfWeapon,
	attribute		=	Hydra_Mk66_Variants["HYDRA_70_M257"].wsTypeOfWeapon,
})

declare_loadout({
	category		=	CAT_ROCKETS,
	CLSID			=	"{M259_HYDRA}",
	Count			=	1,
	Elements		=	{{ ShapeName = Hydra_Mk66_Variants["HYDRA_70_M259"].model }},
	Weight			=	Hydra_Mk66_Variants["HYDRA_70_M259"].fm.mass,
	wsTypeOfWeapon	=	Hydra_Mk66_Variants["HYDRA_70_M259"].wsTypeOfWeapon,
	attribute		=	Hydra_Mk66_Variants["HYDRA_70_M259"].wsTypeOfWeapon,
})

declare_loadout({
	category		=	CAT_ROCKETS,
	CLSID			=	"{M274_HYDRA}",
	Count			=	1,
	Elements		=	{{ ShapeName = Hydra_Mk66_Variants["HYDRA_70_M274"].model }},
	Weight			=	Hydra_Mk66_Variants["HYDRA_70_M274"].fm.mass,
	wsTypeOfWeapon	=	Hydra_Mk66_Variants["HYDRA_70_M274"].wsTypeOfWeapon,
	attribute		=	Hydra_Mk66_Variants["HYDRA_70_M274"].wsTypeOfWeapon,
})

BatchDeclareLoadout(
	RLData.M260,
	{
		["{M260_M151}"]	= { weaponResName = "HYDRA_70_M151" },
		["{M260_M156}"]	= { weaponResName = "HYDRA_70_M156" },
		["{M260_M229}"]	= { weaponResName = "HYDRA_70_M229" },
		["{M260_M257}"]	= { weaponResName = "HYDRA_70_M257" },
		["{M260_M259}"]	= { weaponResName = "HYDRA_70_M259" },
		["{M260_M274}"]	= { weaponResName = "HYDRA_70_M274" },
		["{M260_A_M151_B_M156}"] = {
			multiWeaponTable	= {
				["HYDRA_70_M151"] = 4,
				["HYDRA_70_M156"] = 3,
			},
			Elements			= fillZonedLauncher(RLData.M260, { ["A"] = "{M151_HYDRA}", ["B"] = "{M156_HYDRA}" }),
			displayNamePstfx	= ", " .. LocStr.PodZones .. ": A - M151; B - M156",
			wsTypeOfWeapon		= "weapons.nurs.HYDRA_70_M151"
		},
		["{M260_A_M151_B_M257}"] = {
			multiWeaponTable	= {
				["HYDRA_70_M151"] = 4,
				["HYDRA_70_M257"] = 3,
			},
			Elements			= fillZonedLauncher(RLData.M260, { ["A"] = "{M151_HYDRA}", ["B"] = "{M257_HYDRA}" }),
			displayNamePstfx	= ", " .. LocStr.PodZones .. ": A - M151; B - M257",
			wsTypeOfWeapon		= "weapons.nurs.HYDRA_70_M151"
		},
		["{M260_A_M151_B_M259}"] = {
			multiWeaponTable	= {
				["HYDRA_70_M151"] = 4,
				["HYDRA_70_M259"] = 3,
			},
			Elements			= fillZonedLauncher(RLData.M260, { ["A"] = "{M151_HYDRA}", ["B"] = "{M259_HYDRA}" }),
			displayNamePstfx	= ", " .. LocStr.PodZones .. ": A - M151; B - M259",
			wsTypeOfWeapon		= "weapons.nurs.HYDRA_70_M151"
		},
		["{M260_A_M151_B_M274}"] = {
			multiWeaponTable	= {
				["HYDRA_70_M151"] = 4,
				["HYDRA_70_M274"] = 3,
			},
			Elements			= fillZonedLauncher(RLData.M260, { ["A"] = "{M151_HYDRA}", ["B"] = "{M274_HYDRA}" }),
			displayNamePstfx	= ", " .. LocStr.PodZones .. ": A - M151; B - M274",
			wsTypeOfWeapon		= "weapons.nurs.HYDRA_70_M151"
		},
		["{M260_A_M229_B_M156}"] = {
			multiWeaponTable	= {
				["HYDRA_70_M229"] = 4,
				["HYDRA_70_M156"] = 3,
			},
			Elements			= fillZonedLauncher(RLData.M260, { ["A"] = "{M229_HYDRA}", ["B"] = "{M156_HYDRA}" }),
			displayNamePstfx	= ", " .. LocStr.PodZones .. ": A - M229; B - M156",
			wsTypeOfWeapon		= "weapons.nurs.HYDRA_70_M229"
		},
		["{M260_A_M229_B_M257}"] = {
			multiWeaponTable	= {
				["HYDRA_70_M229"] = 4,
				["HYDRA_70_M257"] = 3,
			},
			Elements			= fillZonedLauncher(RLData.M260, { ["A"] = "{M229_HYDRA}", ["B"] = "{M257_HYDRA}" }),
			displayNamePstfx	= ", " .. LocStr.PodZones .. ": A - M229; B - M257",
			wsTypeOfWeapon		= "weapons.nurs.HYDRA_70_M229"
		},
		["{M260_A_M229_B_M259}"] = {
			multiWeaponTable	= {
				["HYDRA_70_M229"] = 4,
				["HYDRA_70_M259"] = 3,
			},
			Elements			= fillZonedLauncher(RLData.M260, { ["A"] = "{M229_HYDRA}", ["B"] = "{M259_HYDRA}" }),
			displayNamePstfx	= ", " .. LocStr.PodZones .. ": A - M229; B - M259",
			wsTypeOfWeapon		= "weapons.nurs.HYDRA_70_M229"
		},
	},
	Hydra_Mk66_Variants
)

BatchDeclareLoadout(
	RLData.M261,
	{
		-- ["M261_MK151"]			= { weaponResName = "HYDRA_70_M151" },
		-- ["{M261_M151_M433}"]	= { weaponResName = "HYDRA_70_M151_M433" },
		["M261_MK156"]			= { weaponResName = "HYDRA_70_M156" },
		["{M261_M229}"]			= { weaponResName = "HYDRA_70_M229" },
		["{M261_M257}"]			= { weaponResName = "HYDRA_70_M257" },
		["{M261_M259}"]			= { weaponResName = "HYDRA_70_M259" },
		["{M261_M274}"]			= { weaponResName = "HYDRA_70_M274" },
		["{M261_M282}"]			= { weaponResName = "HYDRA_70_M282" },
		["{M261_OUTBOARD_AB_M151_E_M274}"] = {
			multiWeaponTable	= {
				["HYDRA_70_M151"] = 16,
				["HYDRA_70_M274"] = 3,
			},
			Elements			= fillZonedLauncher(RLData.M261, { ["A"] = "{M151_HYDRA}", ["B"] = "{M151_HYDRA}", ["E"] = "{M274_HYDRA}" }),
			displayNamePstfx	= ", " .. LocStr.PodZones .. ": A/B - M151; E - M274",
			wsTypeOfWeapon		= "weapons.nurs.HYDRA_70_M151"
		},
		["{M261_OUTBOARD_AB_M151_E_M257}"] = {
			multiWeaponTable	= {
				["HYDRA_70_M151"] = 16,
				["HYDRA_70_M257"] = 3,
			},
			Elements			= fillZonedLauncher(RLData.M261, { ["A"] = "{M151_HYDRA}", ["B"] = "{M151_HYDRA}", ["E"] = "{M257_HYDRA}" }),
			displayNamePstfx	= ", " .. LocStr.PodZones .. ": A/B - M151; E - M257",
			wsTypeOfWeapon		= "weapons.nurs.HYDRA_70_M151"
		},
		["{M261_INBOARD_DE_M151_C_M274}"] = {
			multiWeaponTable	= {
				["HYDRA_70_M151"] = 16,
				["HYDRA_70_M274"] = 3,
			},
			Elements			= fillZonedLauncher(RLData.M261, { ["C"] = "{M274_HYDRA}", ["D"] = "{M151_HYDRA}", ["E"] = "{M151_HYDRA}" }),
			displayNamePstfx	= ", " .. LocStr.PodZones .. ": C - M274; D/E - M151",
			wsTypeOfWeapon		= "weapons.nurs.HYDRA_70_M151"
		},
		["{M261_INBOARD_DE_M151_C_M257}"] = {
			multiWeaponTable	= {
				["HYDRA_70_M151"] = 16,
				["HYDRA_70_M257"] = 3,
			},
			Elements			= fillZonedLauncher(RLData.M261, { ["C"] = "{M257_HYDRA}", ["D"] = "{M151_HYDRA}", ["E"] = "{M151_HYDRA}" }),
			displayNamePstfx	= ", " .. LocStr.PodZones .. ": C - M257; D/E - M151",
			wsTypeOfWeapon		= "weapons.nurs.HYDRA_70_M151"
		},
	},
	Hydra_Mk66_Variants
)

--------- START MODED APKWS MPP ---------
-----------------------------------------

local MODED_AGR_20_M282 =
{
	category		= CAT_MISSILES,
	name			= "MODED_AGR_20_M282",
	user_name		= _("HYDRA-70 MPP APKWS"),
	scheme			= "APKWS",
	class_name		= "wAmmunitionLaserHoming",
	model			= "AGR_20",
	wsTypeOfWeapon 	= {wsType_Weapon, wsType_Missile, wsType_AS_Missile, WSTYPE_PLACEHOLDER},
	
	mass			= 17,
	Escort			= 1,
	Head_Type		= 4,
	sigma			= {1, 1, 1},
	H_max			= 5000.0,
	H_min			= 1.0,
	Diam			= 70.0,
	Cx_pil			= 0.00244140625,
	D_max			= 5000.0,
	D_min			= 1000.0,
	Head_Form		= 1,
	Life_Time		= 90.0,
	Nr_max			= 25,
	v_min			= 140.0,
	v_mid			= 350.0,
	Mach_max		= 3.0,
	t_b				= 0.0,
	t_acc			= 0.0,
	t_marsh			= 1.1,
	Range_max		= 7500.0,
	H_min_t			= 15.0,
	Fi_start		= 0.4,
	Fi_rak			= 3.14152,
	Fi_excort		= 1.05,
	Fi_search		= 0.7,
	OmViz_max		= 0.35,
	exhaust			= {0.78, 0.78, 0.78, 0.3};
	X_back			= -2.0,
	Y_back			= -0.0,
	Z_back			= 0.0,
	Reflection		= 0.04,
	KillDistance	= 0.0,
	M				= 15,
	
	warhead		= predefined_warhead("HYDRA_70_M282"),
	
	shape_table_data =
	{
		{
			name  = "MODED_AGR_20_M282";
			file  = "AGR_20";
			life  = 1;
			fire  = { 0, 1};
			username = _("APKWS"),
			index 	 = WSTYPE_PLACEHOLDER,
		},
	},
	
	march = {
		fuel_mass   			= 3.175,
		impulse     			= 230,
		boost_time  			= 0,
		work_time   			= 1.1,
		boost_factor			= 1,
		nozzle_position 	    = {{-0.95, 0, 0}},
		nozzle_orientationXYZ   = {{0, 0, 0}},
		tail_width  			= 0.052,
		boost_tail  			= 1,
		work_tail   			= 1,
		smoke_color				= {0.5, 0.474, 0.443},
		smoke_transparency		= 0.5,
		custom_smoke_dissipation_factor		= 0.2,		
	},

	fm = {
		mass        = 17,  
		caliber     = 0.07, 
		cx_coeff    = {1, 1.5, 0.68, 0.7, 1.75},
		L           = 1.81,
		I           = 1 / 12 * 15 * 1.81 * 1.81,
		Ma          = 0.84,
		Mw          = 3.5,
		shapeName   = "AGR_20",
		wind_sigma	= 5.0,
		wind_time	= 1.1,
		dCydA		= {0.11, 0.11},
		A			= 0.36,
		Sw			= 0.04,
		maxAoa		= math.rad(8),		
		finsTau		= 0.1,
		Mx0			= 0.2,
		Ma_x		= 1.8,
		Mw_x		= 1.12,
		I_x			= 1.01,
	},
	
	seeker = {
		delay				= 1.0,
		FOV					= math.rad(40),
		max_seeker_range	= 20000,
	},

	autopilot = {
		delay				= 0.9,
		op_time				= 45,
		dV_dt				= 20,
		Knav				= 12,
		Tf					= 0.2,
		Kd					= 150,
		Kdx					= 0.2,
		Kix					= 0.8,
		gload_limit			= 10,
		fins_limit			= math.rad(20),
		fins_limit_x		= math.rad(10),
	},
}

declare_weapon(MODED_AGR_20_M282)

declare_loadout({
	category 			= CAT_ROCKETS,
	CLSID	 			= "M261_MK151",
	Picture				= "LAU61.png",
	--PictureBlendColor	= "0xffffffff",
	Cx_pil				= 0.00146484375,
	displayName			= _("M261 - 19 x MPP APKWS, Laser Guided Rkts M282"),
	Count				= 19,
	Weight				= 29.5 + 19*17,
	wsTypeOfWeapon		= MODED_AGR_20_M282.wsTypeOfWeapon,
	attribute			= {4,	4,	32,	WSTYPE_PLACEHOLDER},
	Elements			= ContainerWithSubshapes("M261", "AGR_20", 19, "tube_%d")
})

declare_loadout({
	category 			= CAT_ROCKETS,
	CLSID	 			= "{BRU33_LAU61}",
	Picture				= "LAU61.png",
	--PictureBlendColor	= "0xffffffff",
	Cx_pil				= 0.00146484375,
	displayName			= _("LAU-61 pod - 19 x MPP APKWS, Laser Guided Rkts M282"),
	Count				= 19,
	Weight				= 29.5 + 19*17,
	wsTypeOfWeapon		= MODED_AGR_20_M282.wsTypeOfWeapon,
	attribute			= {4,	4,	32,	WSTYPE_PLACEHOLDER},
	Elements			= ContainerWithSubshapes("LAU-61", "AGR_20", 19)
})

--------- END MODED APKWS MPP ---------
---------------------------------------

--------- START MODED APKWS HE ----------
-----------------------------------------

local MODED_AGR_20_M151 =
{
	category		= CAT_MISSILES,
	name			= "MODED_AGR_20_M151",
	user_name		= _("HYDRA-70 HE APKWS"),
	scheme			= "APKWS",
	class_name		= "wAmmunitionLaserHoming",
	model			= "AGR_20",
	wsTypeOfWeapon 	= {wsType_Weapon, wsType_Missile, wsType_AS_Missile, WSTYPE_PLACEHOLDER},
	
	mass			= 15,
	Escort			= 1,
	Head_Type		= 4,
	sigma			= {1, 1, 1},
	H_max			= 5000.0,
	H_min			= 1.0,
	Diam			= 70.0,
	Cx_pil			= 0.00244140625,
	D_max			= 5000.0,
	D_min			= 1000.0,
	Head_Form		= 1,
	Life_Time		= 90.0,
	Nr_max			= 25,
	v_min			= 140.0,
	v_mid			= 350.0,
	Mach_max		= 3.0,
	t_b				= 0.0,
	t_acc			= 0.0,
	t_marsh			= 1.1,
	Range_max		= 5000.0,
	H_min_t			= 15.0,
	Fi_start		= 0.4,
	Fi_rak			= 3.14152,
	Fi_excort		= 1.05,
	Fi_search		= 0.7,
	OmViz_max		= 0.35,
	exhaust			= {0.78, 0.78, 0.78, 0.3};
	X_back			= -2.0,
	Y_back			= -0.0,
	Z_back			= 0.0, -- -0.1,
	Reflection		= 0.01,
	KillDistance	= 0.0,
	M				= 15,
	
	warhead		= predefined_warhead("HYDRA_70_M151"),
	
	shape_table_data =
	{
		{
			name  = "MODED_AGR_20_M151";
			file  = "AGR_20";
			life  = 1;
			fire  = { 0, 1};
			username = _("APKWS"),
			index 	 = WSTYPE_PLACEHOLDER,
		},
	},
	
	march = {
		fuel_mass   			= 3.175,
		impulse     			= 210,
		boost_time  			= 0,
		work_time   			= 1.1,
		boost_factor			= 1,
		nozzle_position 	    = {{-0.95, 0, 0}},
		nozzle_orientationXYZ   = {{0, 0, 0}},
		tail_width  			= 0.052,
		boost_tail  			= 1,
		work_tail   			= 1,
		smoke_color				= {0.5, 0.474, 0.443},
		smoke_transparency		= 0.5,
		custom_smoke_dissipation_factor		= 0.2,		
	},

	fm = {
		mass        = 15,  
		caliber     = 0.07, 
		cx_coeff    = {1, 1.5, 0.68, 0.7, 1.75},
		L           = 1.81,
		I           = 1 / 12 * 15 * 1.81 * 1.81,
		Ma          = 0.84,
		Mw          = 3.5,
		shapeName   = "AGR_20",
		wind_sigma	= 5.0,
		wind_time	= 1.1,
		dCydA		= {0.11, 0.11},
		A			= 0.36,
		Sw			= 0.04,
		maxAoa		= math.rad(8),		
		finsTau		= 0.1,
		Mx0			= 0.2,
		Ma_x		= 1.8,
		Mw_x		= 1.12,
		I_x			= 1.01,
	},
	
	seeker = {
		delay				= 1.1,
		FOV					= math.rad(40),
		max_seeker_range	= 13000,
	},

	autopilot = {
		delay				= 0.9,
		op_time				= 45,
		dV_dt				= 20,
		Knav				= 12,
		Tf					= 0.2,
		Kd					= 150,
		Kdx					= 0.2,
		Kix					= 0.8,
		gload_limit			= 10,
		fins_limit			= math.rad(20),
		fins_limit_x		= math.rad(10),
	},
}

declare_weapon(MODED_AGR_20_M151)

declare_loadout({
	category 			= CAT_ROCKETS,
	CLSID	 			= "{M261_M151_M433}",
	Picture				= "LAU61.png",
	--PictureBlendColor	= "0xffffffff",
	Cx_pil				= 0.00146484375,
	displayName			= _("M261 - 19 x HE APKWS, Laser Guided Rkts M151"),
	Count				= 19,
	Weight				= 29.5 + 19*15,
	wsTypeOfWeapon		= MODED_AGR_20_M151.wsTypeOfWeapon,
	attribute			= {4,	4,	32,	WSTYPE_PLACEHOLDER},
	Elements			= ContainerWithSubshapes("M261", "AGR_20", 19, "tube_%d")
})

--------- END MODED APKWS HE ----------
---------------------------------------


declare_loadout({
	category		=	CAT_ROCKETS,
	CLSID			=	"{UB-32A-24}",
	Picture			=	"rus_UB-32_S-5KO.png",
	PictureBlendColor = "0xffffffff",
	Cx_pil			=	0.00196533203125,
	displayName		=	_("UB-32A-24 pod - 32 x S-5KO"),
	Count			=	32,
	Elements		=	ContainerWithSubshapes("UB_32A_24", "S-5KO", 32, "tube_%d"),
	Weight			=	275,
	wsTypeOfWeapon	=	{4,	7,	33,	31},
	attribute		=	{4,	7,	32,	WSTYPE_PLACEHOLDER},
})

local ARF8M3_Variants = BatchDeclareRockets({
	commonParams = {
		baseFM			= predefined_fm("ARF8M3"),
		baseEngine		= predefined_engine("ARF8M3"),
		properties		= { dist_min = 450, dist_max = 4500 }
	},
	variantSpecificParams = {
		["ARF8M3API"] = {
			displayName		= _("ARF-8/M3 API"),
			displayNameShort= "ARF-8",
			schemeName		= "nurs-standard",
			payloadParams	= { warhead = predefined_warhead("ARF8M3API")},
			modelFileName	= "ARF8M3_API",
		},
		["ARF8M3HEI"] = {
			displayName		= _("ARF-8/M3 HEI Heavy"),
			displayNameShort= "ARF-8",
			schemeName		= "nurs-standard",
			payloadParams	= { warhead = predefined_warhead("ARF8M3HEIHeavy")},
			modelFileName	= "ARF8M3_HEI",
		},
		["ARF8M3TPSM"] = {
			displayName		= _("ARF-8/M3 TP-SM"),
			displayNameShort= "ARF-8",
			schemeName		= "nurs-marker",
			payloadParams	= { warhead = predefined_warhead("ARF8M3TPSM")},
			modelFileName	= "ARF8M3_TP",
		},
	}
})

BatchDeclareLoadout(
	RLData.LR25,
	{
		["{LR25_ARF8M3_API}"]	= { weaponResName = "ARF8M3API", },
		["{LR25_ARF8M3_HEI}"]	= { weaponResName = "ARF8M3HEI", },
		["{LR25_ARF8M3_TPSM}"]	= { weaponResName = "ARF8M3TPSM", },
	},
	ARF8M3_Variants
)

local S_25_O = {
	category		= CAT_ROCKETS,
	name			= "S-25-O",
	user_name		= _("S-25-O"),
	display_name_short	= _("S-25"),
	scheme			= "nurs-standard",
	class_name		= "wAmmunitionNURS",
	model			= "C-25O",
	wsTypeOfWeapon 	= {wsType_Weapon, wsType_NURS, wsType_Rocket, WSTYPE_PLACEHOLDER},

	fm = {
		mass        = 380,
		caliber     = 0.42, 
		cx_coeff    = {1,0.8459661,0.67,0.3019448,2.08},
		L           = 3.31,
		I           = 433.01467,
		Ma          = 0.192352,
		Mw          = 2.521688,
		shapeName   = "C-25O",
		wind_time   = 2.2,
		wind_sigma  = 3,
	},

	engine = predefined_engine("9GSch605"),

	warhead = predefined_warhead("S_25_O"),

	properties = {
		dist_min = 1600,
		dist_max = 3000
	},

	shape_table_data = {{
		name  = "S-25-O";
		file  = "C-25O";
		username = _("S-25-O Frag"),
		index 	 = WSTYPE_PLACEHOLDER,
	}},
}
declare_weapon(S_25_O)

rockets_data["S_25_O"] = {name = _("S-25-O - 420mm UnGd Rkt, 380kg Frag"), payload_CLSID = "{S_25_O}", mass = S_25_O.fm.mass + 65, mass_empty = 65, wsType = S_25_O.wsTypeOfWeapon, Cx = 0.001708984375, picture = "S25_O.png"}

declare_loadout({
	category		= CAT_ROCKETS,
	CLSID			= rockets_data["S_25_O"].payload_CLSID,
	Picture			= rockets_data["S_25_O"].picture,
	Cx_pil			= rockets_data["S_25_O"].Cx,
	displayName		= rockets_data["S_25_O"].name,
	Count			= 1,
	Elements		= {
		{ ShapeName = "C-25PU", IsAdapter = true, DrawArgs = {{3, 0.5}}},
		{ ShapeName = "C-25O", DrawArgs = {{2, 1}}, connector_name = "tube_1"}
	},
	Weight			= rockets_data["S_25_O"].mass,
	Weight_Empty	= rockets_data["S_25_O"].mass_empty,
	wsTypeOfWeapon	= rockets_data["S_25_O"].wsType,
	attribute		= {wsType_Weapon, wsType_NURS, wsType_Container, WSTYPE_PLACEHOLDER},
})

declare_loadout({
	category		= CAT_ROCKETS,
	CLSID			= "{TWIN_S25_O}",
	Picture			= rockets_data["S_25_O"].picture,
	Cx_pil			= 0.0004 + 2 * rockets_data["S_25_O"].Cx,
	displayName		= "2 x " .. rockets_data["S_25_O"].name,
	Count			= 2,
	JettisonSubmunitionOnly = true,
	Elements		= {
		{ ShapeName = "su-27-twinpylon", IsAdapter = true },
		{ payload_CLSID = rockets_data["S_25_O"].payload_CLSID, connector_name = "S-25-L" },
		{ payload_CLSID = rockets_data["S_25_O"].payload_CLSID, connector_name = "S-25-R" },
	},
	Weight			= 32 + 2 * rockets_data["S_25_O"].mass,
	Weight_Empty	= 32 + 2 * rockets_data["S_25_O"].mass_empty,
	wsTypeOfWeapon	= rockets_data["S_25_O"].wsType,
	attribute		= {wsType_Weapon, wsType_NURS, wsType_Container, WSTYPE_PLACEHOLDER},
})

local SNEB68_F1B_Variants = BatchDeclareRockets({
	commonParams = {
		baseFM			= predefined_fm("SNEB68"),
		baseEngine		= predefined_engine("SNEBType25F1B"),
		modelFileName	= "SA342_SNEB68"
	},
	variantSpecificParams = {
		["SNEB_TYPE250_F1B"] = {
			displayName		= _("SNEB Type 250 F1B TP-SM"),
			displayNameShort= "SNEB250",
			schemeName		= "nurs-marker",
			payloadParams	= { warhead = predefined_warhead("SNEBType20") },
		},
		["SNEB_TYPE251_F1B"] = {
			displayName		= _("SNEB Type 251 F1B HE"),
			displayNameShort= "SNEB251",
			schemeName		= "nurs-standard",
			payloadParams	= { warhead = predefined_warhead("SNEBType21") },
		},
		["SNEB_TYPE252_F1B"] = {
			displayName		= _("SNEB Type 252 F1B TP"),
			displayNameShort= "SNEB252",
			schemeName		= "nurs-standard",
			payloadParams	= { warhead = predefined_warhead("SNEBType22") },
		},
		["SNEB_TYPE253_F1B"] = {
			displayName		= _("SNEB Type 253 F1B HEAT"),
			displayNameShort= "SNEB253",
			schemeName		= "nurs-standard",
			payloadParams	= { warhead = predefined_warhead("SNEBType23") },
		},
		["SNEB_TYPE254_F1B_RED"] = {
			displayName		= _("SNEB Type 254 F1B SM Red"),
			displayNameShort= "SNEB254",
			schemeName		= "nurs-marker",
			payloadParams	= { warhead = predefined_warhead("SNEBType24Red") },
		},
		["SNEB_TYPE254_F1B_YELLOW"] = {
			displayName		= _("SNEB Type 254 F1B SM Yellow"),
			displayNameShort= "SNEB254",
			schemeName		= "nurs-marker",
			payloadParams	= { warhead = predefined_warhead("SNEBType24Yellow") },
		},
		["SNEB_TYPE254_F1B_GREEN"] = {
			displayName		= _("SNEB Type 254 F1B SM Green"),
			displayNameShort= "SNEB254",
			schemeName		= "nurs-marker",
			payloadParams	= { warhead = predefined_warhead("SNEBType24Green") },
		},
		["SNEB_TYPE256_F1B"] = {
			displayName		= _("SNEB Type 256 F1B HE/Frag"),
			displayNameShort= "SNEB256",
			schemeName		= "nurs-standard",
			payloadParams	= { warhead = predefined_warhead("SNEBType26") },
		},
		["SNEB_TYPE257_F1B"] = {
			displayName		= _("SNEB Type 257 F1B HE/Frag Lg Whd"),
			displayNameShort= "SNEB257",
			schemeName		= "nurs-standard",
			payloadParams	= { warhead = predefined_warhead("SNEBType27") },
		},
		["SNEB_TYPE259E_F1B"] = {
			displayName		= _("SNEB Type 259E F1B IL"),
			displayNameShort= "SNEB259",
			schemeName		= "nurs-light",
			payloadParams	= { launcher = { ammunition = S_8OM_FLARE, delay = 10.1 } },
		},
	}
})

local SNEB68_Rotary_Variants = BatchDeclareRockets({
	commonParams = {
		baseFM			= predefined_fm("SNEB68"),
		baseEngine		= predefined_engine("SNEBType25H1"),
		modelFileName	= "SA342_SNEB68",
	},
	variantSpecificParams = {
		["SNEB_TYPE251_H1"] = {
			displayName		= _("SNEB Type 251 H1 HE"),
			displayNameShort= "SNEB251",
			schemeName		= "nurs-standard",
			payloadParams	= { warhead = predefined_warhead("SNEBType21") },
		},
		["SNEB_TYPE252_H1"] = {
			displayName		= _("SNEB Type 252 H1 TP"),
			displayNameShort= "SNEB252",
			schemeName		= "nurs-standard",
			payloadParams	= { warhead = predefined_warhead("SNEBType22") },
		},
		["SNEB_TYPE253_H1"] = {
			displayName		= _("SNEB Type 253 H1 HEAT"),
			displayNameShort= "SNEB253",
			schemeName		= "nurs-standard",
			payloadParams	= { warhead = predefined_warhead("SNEBType23") },
		},
		["SNEB_TYPE254_H1_RED"] = {
			displayName		= _("SNEB Type 254 H1 SM Red"),
			displayNameShort= "SNEB254",
			schemeName		= "nurs-marker",
			payloadParams	= { warhead = predefined_warhead("SNEBType24Red") },
		},
		["SNEB_TYPE254_H1_YELLOW"] = {
			displayName		= _("SNEB Type 254 H1 SM Yellow"),
			displayNameShort= "SNEB254",
			schemeName		= "nurs-marker",
			payloadParams	= { warhead = predefined_warhead("SNEBType24Yellow") },
		},
		["SNEB_TYPE254_H1_GREEN"] = {
			displayName		= _("SNEB Type 254 H1 SM Green"),
			displayNameShort= "SNEB254",
			schemeName		= "nurs-marker",
			payloadParams	= { warhead = predefined_warhead("SNEBType24Green") },
		},
		["SNEB_TYPE256_H1"] = {
			displayName		= _("SNEB Type 256 H1 HE/Frag"),
			displayNameShort= "SNEB256",
			schemeName		= "nurs-standard",
			payloadParams	= { warhead = predefined_warhead("SNEBType26") },
		},
		["SNEB_TYPE257_H1"] = {
			displayName		= _("SNEB Type 257 H1 HE/Frag Lg Whd"),
			displayNameShort= "SNEB257",
			schemeName		= "nurs-standard",
			payloadParams	= { warhead = predefined_warhead("SNEBType27") },
		},
		["SNEB_TYPE259E_H1"] = {
			displayName		= _("SNEB Type 259E H1 IL"),
			displayNameShort= "SNEB259",
			schemeName		= "nurs-light",
			payloadParams	= { launcher = { ammunition = S_8OM_FLARE, delay = 10.1 } },
		},
	}
})

BatchDeclareLoadout(
	RLData.MATRAF1,
	{
		["{MATRA_F1_SNEBT250}"]			= { weaponResName = "SNEB_TYPE250_F1B" },
		["{MATRA_F1_SNEBT251}"]			= { weaponResName = "SNEB_TYPE251_F1B" },
		["{MATRA_F1_SNEBT252}"]			= { weaponResName = "SNEB_TYPE252_F1B" },
		["{MATRA_F1_SNEBT253}"]			= { weaponResName = "SNEB_TYPE253_F1B" },
		["{MATRA_F1_SNEBT254_RED}"]		= { weaponResName = "SNEB_TYPE254_F1B_RED" },
		["{MATRA_F1_SNEBT254_YELLOW}"]	= { weaponResName = "SNEB_TYPE254_F1B_YELLOW" },
		["{MATRA_F1_SNEBT254_GREEN}"]	= { weaponResName = "SNEB_TYPE254_F1B_GREEN" },
		["{MATRA_F1_SNEBT256}"]			= { weaponResName = "SNEB_TYPE256_F1B" },
		["{MATRA_F1_SNEBT257}"]			= { weaponResName = "SNEB_TYPE257_F1B" },
		["{MATRA_F1_SNEBT259E}"]		= { weaponResName = "SNEB_TYPE259E_F1B" },
	},
	SNEB68_F1B_Variants
)

BatchDeclareLoadout(
	RLData.MATRAF4,
	{
		["{MATRA_F4_SNEBT250}"]			= { weaponResName = "SNEB_TYPE250_F1B" },
		["{MATRA_F4_SNEBT251}"]			= { weaponResName = "SNEB_TYPE251_F1B" },
		["{MATRA_F4_SNEBT252}"]			= { weaponResName = "SNEB_TYPE252_F1B" },
		["{MATRA_F4_SNEBT253}"]			= { weaponResName = "SNEB_TYPE253_F1B" },
		["{MATRA_F4_SNEBT254_RED}"]		= { weaponResName = "SNEB_TYPE254_F1B_RED" },
		["{MATRA_F4_SNEBT254_YELLOW}"]	= { weaponResName = "SNEB_TYPE254_F1B_YELLOW" },
		["{MATRA_F4_SNEBT254_GREEN}"]	= { weaponResName = "SNEB_TYPE254_F1B_GREEN" },
		["{MATRA_F4_SNEBT256}"]			= { weaponResName = "SNEB_TYPE256_F1B" },
		["{MATRA_F4_SNEBT257}"]			= { weaponResName = "SNEB_TYPE257_F1B" },
		["{MATRA_F4_SNEBT259E}"]		= { weaponResName = "SNEB_TYPE259E_F1B" },
	},
	SNEB68_F1B_Variants
)

-- Type 250 is available only with F1B motor
SNEB68_Rotary_Variants["SNEB_TYPE250_F1B"] = SNEB68_F1B_Variants["SNEB_TYPE250_F1B"]

BatchDeclareLoadout(
	RLData.TELSON8,
	{
		["{TELSON8_SNEBT250}"]			= { weaponResName = "SNEB_TYPE250_F1B", Picture = "SNEB68_Smoke.png" },
		["{TELSON8_SNEBT251}"]			= { weaponResName = "SNEB_TYPE251_H1" },
		["{TELSON8_SNEBT252}"]			= { weaponResName = "SNEB_TYPE252_H1" },
		["{TELSON8_SNEBT253}"]			= { weaponResName = "SNEB_TYPE253_H1" },
		["{TELSON8_SNEBT254_RED}"]		= { weaponResName = "SNEB_TYPE254_H1_RED", Picture = "SNEB68_Smoke.png", PictureBlendColor = "0xff0000ff" },
		["{TELSON8_SNEBT254_YELLOW}"]	= { weaponResName = "SNEB_TYPE254_H1_YELLOW", Picture = "SNEB68_Smoke.png", PictureBlendColor = "0xffff00ff" },
		["{TELSON8_SNEBT254_GREEN}"]	= { weaponResName = "SNEB_TYPE254_H1_GREEN", Picture = "SNEB68_Smoke.png", PictureBlendColor = "0x00ff00ff" },
		["{TELSON8_SNEBT256}"]			= { weaponResName = "SNEB_TYPE256_H1" },
		["{TELSON8_SNEBT257}"]			= { weaponResName = "SNEB_TYPE257_H1" },
		["{TELSON8_SNEBT259E}"]			= { weaponResName = "SNEB_TYPE259E_H1" },
		-- Backwards-compatibility section
		["{LAU_SNEB68_WP}"]				= { weaponResName = "SNEB_TYPE250_F1B", Picture = "SNEB68_Smoke.png" },
		["{LAU_SNEB68G}"]				= { weaponResName = "SNEB_TYPE253_H1" },
	},
	SNEB68_Rotary_Variants
)

local S8_FFAR_Variants = BatchDeclareRockets({
	commonParams = {
		baseFM			= predefined_fm("S8"),
		baseEngine		= predefined_engine("9GSch4421"),
		properties		= {
			dist_min	= 600,
			dist_max	= 3000
		}
	},
	variantSpecificParams = {
		["C_8"] = {
			displayName		= _("S-8KOM HEAT/Frag"),
			displayNameShort= _("S-8"),
			schemeName		= "nurs-standard",
			modelFileName	= "s-8_kom",
			payloadParams	= { warhead = predefined_warhead("S_8KOM") },
			paramOverrides	= { wsTypeOfWeaponL4 = C_8 }
		},
		["C_8OM"] = {
			displayName		= _("S-8OM IL"),
			displayNameShort= _("S-8"),
			schemeName		= "nurs-light",
			modelFileName	= "s-8_om",
			payloadParams	= { launcher = { ammunition = S_8OM_FLARE, delay = 15 } },
			paramOverrides	= { wsTypeOfWeaponL4 = C_8OM },
			properties		= {
				dist_min	= 4000,
				dist_max	= 4500,
				max_TOF		= 15.1 -- exaggerated a bit to avoid AI overshooting the target due to non-instant deceleration of the flare element
			}
		},
		["C_8CM"] = {
			displayName		= _("S-8TsM SM Orange"),
			displayNameShort = _("S-8"),
			schemeName		= "nurs-marker",
			modelFileName	= "s-8_sm",
			payloadParams	= { warhead = predefined_warhead("S_8TsM_ORANGE") },
			paramOverrides	= { wsTypeOfWeaponL4 = C_8CM }
		},
		["C_8CM_GN"] = {
			displayName		= _("S-8TsM SM Green"),
			displayNameShort= _("S-8"),
			schemeName		= "nurs-marker",
			modelFileName	= "s-8_sm",
			payloadParams	= { warhead = predefined_warhead("S_8TsM_GREEN") },
			paramOverrides	= { wsTypeOfWeaponL4 = C_8CM_GN }
		},
		["C_8CM_RD"] = {
			displayName		= _("S-8TsM SM Red"),
			displayNameShort= _("S-8"),
			schemeName		= "nurs-marker",
			modelFileName	= "s-8_sm",
			payloadParams	= { warhead = predefined_warhead("S_8TsM_RED") },
			paramOverrides	= { wsTypeOfWeaponL4 = C_8CM_RD }
		},
		["C_8CM_WH"] = {
			displayName		= _("S-8TsM SM White"),
			displayNameShort= _("S-8"),
			schemeName		= "nurs-marker",
			modelFileName	= "s-8_sm",
			payloadParams	= { warhead = predefined_warhead("S_8TsM_WHITE") },
			paramOverrides	= { wsTypeOfWeaponL4 = C_8CM_WH }
		},
		["C_8CM_BU"] = {
			displayName		= _("S-8TsM SM Blue"),
			displayNameShort= _("S-8"),
			schemeName		= "nurs-marker",
			modelFileName	= "s-8_sm",
			payloadParams	= { warhead = predefined_warhead("S_8TsM_BLUE") },
			paramOverrides	= { wsTypeOfWeaponL4 = C_8CM_BU }
		},
		["C_8CM_YE"] = {
			displayName		= _("S-8TsM SM Yellow"),
			displayNameShort= _("S-8"),
			schemeName		= "nurs-marker",
			modelFileName	= "s-8_sm",
			payloadParams	= { warhead = predefined_warhead("S_8TsM_YELLOW") },
			paramOverrides	= { wsTypeOfWeaponL4 = C_8CM_YE }
		},
		["C_8CM_VT"] = {
			displayName		= _("S-8TsM SM Violet"),
			displayNameShort= _("S-8"),
			schemeName		= "nurs-marker",
			modelFileName	= "s-8_sm",
			payloadParams	= { warhead = predefined_warhead("S_8TsM_VIOLET") },
			paramOverrides	= { wsTypeOfWeaponL4 = C_8CM_VT }
		},
	}
})

local S8_WAFAR_Variants = BatchDeclareRockets({
	commonParams = {
		baseFM			= predefined_fm("S8"),
		baseEngine		= predefined_engine("S8_WAFAR"),
		properties		= {
			dist_min	= 600,
			dist_max	= 4000
		}
	},
	variantSpecificParams = {
		["C_8OFP2"] = {
			displayName		= _("S-8OFP2 MPP"),
			displayNameShort= _("S-8"),
			schemeName		= "nurs-standard",
			modelFileName	= "s-8_ofp2",
			payloadParams	= { warhead = predefined_warhead("S_8OFP") },
			paramOverrides	= { wsTypeOfWeaponL4 = C_8OFP2 }
		},
	}
})

local All_S8_Variants = S8_FFAR_Variants

for descName, desc in pairs(S8_WAFAR_Variants) do
	All_S8_Variants[descName] = desc
end

BatchDeclareLoadout(
	RLData.B8V20A,
	{
		["{6A4B9E69-64FE-439a-9163-3A87FB6A4D81}"]	= { weaponResName = "C_8",			Picture = "rus_B-8V20A_S-8KOM.png",		attribute = { wsType_Weapon, wsType_NURS, wsType_Container, B_8V20A }, },
		["B_8V20A_CM"]								= { weaponResName = "C_8CM",		Picture = "rus_B-8V20A_S-8TsM_OG.png",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, B_8V20A_WP }, },
		["B_8V20A_CM_GN"]							= { weaponResName = "C_8CM_GN",		Picture = "rus_B-8V20A_S-8TsM_GN.png",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, B_8V20A_WP }, },
		["B_8V20A_CM_RD"]							= { weaponResName = "C_8CM_RD",		Picture = "rus_B-8V20A_S-8TsM_RD.png",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, B_8V20A_WP }, },
		["B_8V20A_CM_WH"]							= { weaponResName = "C_8CM_WH",		Picture = "rus_B-8V20A_S-8TsM_WH.png",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, B_8V20A_WP }, },
		["B_8V20A_CM_BU"]							= { weaponResName = "C_8CM_BU",		Picture = "rus_B-8V20A_S-8TsM_BU.png",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, B_8V20A_WP }, },
		["B_8V20A_CM_YE"]							= { weaponResName = "C_8CM_YE",		Picture = "rus_B-8V20A_S-8TsM_YE.png",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, B_8V20A_WP }, },
		["B_8V20A_CM_VT"]							= { weaponResName = "C_8CM_VT",		Picture = "rus_B-8V20A_S-8TsM_VT.png",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, B_8V20A_WP }, },
		["B_8V20A_OFP2"]							= { weaponResName = "C_8OFP2",		Picture = "rus_B-8V20A_S-8OFP2.png",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, B_8V20A_OFP2 }, },
		["B_8V20A_OM"]								= { weaponResName = "C_8OM",		Picture = "rus_B-8V20A_S-8OM.png",		attribute = { wsType_Weapon, wsType_NURS, wsType_Container, B_8V20A_OM }, },
	},
	All_S8_Variants
)

BatchDeclareLoadout(
	RLData.B8M1,
	{
		["{F72F47E5-C83A-4B85-96ED-D3E46671EE9A}"]	= { weaponResName = "C_8",			Picture = "rus_B-8M1_S-8KOM.png",		attribute = { wsType_Weapon, wsType_NURS, wsType_Container, B_20 }, },
		["{3DFB7320-AB0E-11d7-9897-000476191836}"]	= { weaponResName = "C_8CM",		Picture = "rus_B-8M1_S-8TsM.png",		attribute = { wsType_Weapon, wsType_NURS, wsType_Container, B_20CM }, },
		["B-8M1 - 20 S-8OFP2"]						= { weaponResName = "C_8OFP2",		Picture = "rus_B-8M1_S-8OFP2.png",		attribute = { wsType_Weapon, wsType_NURS, wsType_Container, B_8M1_OFP2 }, },
		["{B8M1_S8OM}"]								= { weaponResName = "C_8OM",		Picture = "rus_B-8M1_S-8OM.png", },
	},
	All_S8_Variants
)

declare_loadout({
	category				= CAT_ROCKETS,
	CLSID					= "{TWIN_B_8M1_S_8KOM}",
	wsTypeOfWeapon			= "weapons.nurs.C_8",
	attribute				= {4, 7, 32, WSTYPE_PLACEHOLDER},
	Count					= 40,
	Cx_pil					= 0.0004,
	Picture					= "rus_B-8M1_S-8KOM.png",
	PictureBlendColor		= "0xffffffff",
	displayName				= "2 x " .. _("B-8M1") .. " - 40 x " .. LocStr.UnGdRkts .. ", 80 " .. LocStr.mm .. " " .. All_S8_Variants["C_8"].displayName,
	Weight					= 32 + 2 * (RLData.B8M1.Weight_Empty + 20 * All_S8_Variants["C_8"].fm.mass + All_S8_Variants["C_8"].fm.preLaunchAdditivesMass),
	Weight_Empty			= 32 + 2 * RLData.B8M1.Weight_Empty,
	JettisonSubmunitionOnly	= true,
	Elements				= {
		{ShapeName = "su-27-twinpylon",IsAdapter = true},
		{payload_CLSID = "{F72F47E5-C83A-4B85-96ED-D3E46671EE9A}", DrawArgs = {{3,0.5}}, connector_name = "S-25-L"},
		{payload_CLSID = "{F72F47E5-C83A-4B85-96ED-D3E46671EE9A}", DrawArgs = {{3,0.5}}, connector_name = "S-25-R"},
	}
})

declare_loadout({
	category				= CAT_ROCKETS,
	CLSID					= "{TWIN_B_8M1_S_8_OFP2}",
	wsTypeOfWeapon			= "weapons.nurs.C_8OFP2",
	attribute				= {4, 7, 32, WSTYPE_PLACEHOLDER},
	Count					= 40,
	Cx_pil					= 0.0004,
	Picture					= "rus_B-8M1_S-8OFP2.png",
	PictureBlendColor		= "0xffffffff",
	displayName				= "2 x " .. _("B-8M1") .. " - 40 x " .. LocStr.UnGdRkts .. ", 80 " .. LocStr.mm .. " " .. All_S8_Variants["C_8OFP2"].displayName,
	Weight					= 32 + 2 * (RLData.B8M1.Weight_Empty + 20 * All_S8_Variants["C_8OFP2"].fm.mass + All_S8_Variants["C_8OFP2"].fm.preLaunchAdditivesMass),
	Weight_Empty			= 32 + 2 * RLData.B8M1.Weight_Empty,
	JettisonSubmunitionOnly	= true,
	Elements				= {
		{ShapeName = "su-27-twinpylon",IsAdapter = true},
		{payload_CLSID = "B-8M1 - 20 S-8OFP2", DrawArgs = {{3,0.5}},connector_name = "S-25-L"},
		{payload_CLSID = "B-8M1 - 20 S-8OFP2", DrawArgs = {{3,0.5}},connector_name = "S-25-R"},
	}
})

declare_loadout({
	category				= CAT_ROCKETS,
	CLSID					= "{TWIN_B_8M1_S_8TsM}",
	wsTypeOfWeapon			= "weapons.nurs.C_8CM",
	attribute				= {4, 7, 32, WSTYPE_PLACEHOLDER},
	Count					= 40,
	Cx_pil					= 0.0004,
	Picture					= "rus_B-8M1_S-8TsM.png",
	PictureBlendColor		= "0xffffffff",
	displayName				= "2 x " .. _("B-8M1") .. " - 40 x " .. _("UnGd Rkts") .. ", 80 " .. LocStr.mm .. " " .. All_S8_Variants["C_8CM"].displayName,
	Weight					= 32 + 2 * (RLData.B8M1.Weight_Empty + 20 * All_S8_Variants["C_8CM"].fm.mass + All_S8_Variants["C_8CM"].fm.preLaunchAdditivesMass),
	Weight_Empty			= 32 + 2 * RLData.B8M1.Weight_Empty,
	JettisonSubmunitionOnly	= true,
	Elements				= {
		{ShapeName = "su-27-twinpylon",IsAdapter = true},
		{payload_CLSID = "{3DFB7320-AB0E-11d7-9897-000476191836}", DrawArgs = {{3,0.5}}, connector_name = "S-25-L"},
		{payload_CLSID = "{3DFB7320-AB0E-11d7-9897-000476191836}", DrawArgs = {{3,0.5}}, connector_name = "S-25-R"},
	}
})
